//
//  HomeHeader.swift
//  Do It
//
//  Created by Jim Dovey on 2/11/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI
import Combine
import CoreData

struct HomeHeader: View {
    static let layout: [[TodoItemGroup]] = [
        [.today, .scheduled],
        [.all, .overdue],
    ]

    // START:LinkView
    @Environment(\.managedObjectContext) var objectContext

    private func linkView(for group: TodoItemGroup) -> some View {
        let destination = TodoList(group: group)
            .environment(\.managedObjectContext, objectContext)//<label id="code.7.homeheader.pass.context"/>
        return NavigationLink(destination: destination) {
            HeaderItem(group: group)
        }
    }
    // END:LinkView

    // START:UpdateBody
    var body: some View {
        VStack {
            ForEach(Self.layout, id: \.self) { row in
                HStack(spacing: 12) {
                    // START_HIGHLIGHT
                    ForEach(row, id: \.self, content: self.linkView(for:))
                    // END_HIGHLIGHT
                }
            }
        }
    }
    // END:UpdateBody

    struct HeaderItem: View {
        let group: TodoItemGroup
        @State var itemCount: Int = 0

        // START:ItemNewState
        @Environment(\.managedObjectContext) var objectContext
        @State private var countCancellable: AnyCancellable? = nil
        // END:ItemNewState

        // START:ItemBody
        var body: some View {
            VStack(alignment: .leading) {
                // <literal:elide> ... </literal:elide>
                // END:ItemBody
                HStack {
                    group.icon
                    Spacer()
                    Text("\(itemCount)")
                        .foregroundColor(.primary)
                        .font(.system(.title, design: .rounded))
                        .fontWeight(.bold)
                }
                
                Text(group.title)
                    .foregroundColor(.secondary)
                    .font(.system(.subheadline, design: .rounded))
                    .fontWeight(.medium)
                // START:ItemBody
            }
            .padding()
            .background(
                RoundedRectangle(cornerRadius: 15, style: .continuous)
                    .fill(Color(.tertiarySystemBackground))
            )
            // START_HIGHLIGHT
            .onAppear(perform: startWatchingCount)
            .onDisappear(perform: stopWatchingCount)
            // END_HIGHLIGHT
        }
        // END:ItemBody

        // START:ObserveCount
        private func startWatchingCount() {
            guard countCancellable == nil else { return }
            
            let request = group.fetchRequest
            countCancellable = NotificationCenter.default
                .publisher(
                    for: .NSManagedObjectContextDidSave,
                    object: objectContext)
                .receive(on: RunLoop.main)//<label id="code.7.home.header.pub.runloop"/>
                .compactMap { $0.object as? NSManagedObjectContext }
                .tryMap { try $0.count(for: request) }
                .replaceError(with: 0)//<label id="code.7.home.header.pub.error"/>
                .removeDuplicates()
                .assign(to: \.itemCount, on: self)
            
            if let count = try? objectContext.count(for: request) {
                itemCount = count //<label id="code.7.home.header.count.preflight"/>
            }
        }
        
        private func stopWatchingCount() {
            countCancellable = nil//<label id="code.7.home.header.pub.cancel"/>
        }
        // END:ObserveCount
    }
}

struct HomeHeader_Previews: PreviewProvider {
    static var previews: some View {
        ForEach(ColorScheme.allCases, id: \.self) { colorScheme in
            HomeHeader()
                .padding()
                .background(Color(.systemGroupedBackground))
                .colorScheme(colorScheme)
        }
        .previewLayout(.sizeThatFits)
            // START:Preview
        .environment(\.managedObjectContext, PreviewDataStore.shared.viewContext)
        // END:Preview
    }
}
